/*=======================================================*/
/*
 *  Listing 3.  devinit.c - standard c functions for 
 *                          generic device
 *
 *  The procedures in this module are called by assembly
 *  language procedures in the file devent.a.  They
 *  perform 'generic' operations for a device, and call
 *  device-specific procedures in an external device
 *  implementation module.  For this device, the
 *  external module is 'userdev.c', and the functions
 *  in that file are 'usr_xxx(..)'
 *
 *  Placed in the Public Domain by Forest W. Arnold, 1994
*/
/*=======================================================*/

#include "devinit.h"

#include <intuition/intuitionBase.h>
#include <proto/intuition.h>

/*
 *  initialized in this module
*/

struct DosLibrary *DOSBase = NULL;

/*
 *  signal semaphore for the device.  This is used to
 *  lock/unlock access to any global, shared device
 *  data.
*/

Semaphore_t sem;

/*
 *  external device-specific procedures called from
 *  this module
 *
 *  These procedures must be defined to implement specific
 *  device functionality.
*/

extern int  usr_devInit( struct GenericDevice* );
extern void usr_devOpen( long unit, 
                         struct IOStdReq*,
                         unsigned long,
                         struct GenericDevice* );
extern void usr_devClose( struct IOStdReq*,
                          struct GenericDevice* );
extern int usr_devExpunge( struct GenericDevice* );
extern int usr_devBeginIO( struct IOStdReq* );
extern void usr_devAbortIO( struct IOStdReq* );

/*
 *  forward declarations of procedures in this module
*/


BPTR dev_Expunge( struct GenericDevice *dev );


/*=======================================================*/
/*
 *  dev_Init() - device initialization function
 *
 *  For RTF_AUTOINIT,
 *  this procedure is called after the device has been
 *  allocated.  When this procedure returns the device
 *  pointer, the device is added to the system device list.
 *
 *  If it returns NULL, the device is unloaded.
*/
/*=======================================================*/

struct Device *dev_Init( APTR *dev, BPTR segptr )
{
   struct GenericDevice *device = 
                           (struct GenericDevice*)dev;

   /*
    *  init the device-specific data elements
    *  save the seglist pointer for use in close/expunge
   */

   device->dev_Pad           = 0;
   device->dev_Data.segList  = segptr;
   device->dev_Data.execBase = SysBase;
   device->dev_Data.dbgFH    = NULL;
   device->dev_Data.data     = NULL;

   /*
    *  init the device semaphore
   */

   InitSemaphore(&sem);
   device->dev_Data.lock = &sem;

   /*
    *  open the dos library
   */

   DOSBase = (struct DosLibrary*)
                  OpenLibrary("dos.library",0L);

   /*
    *  call the device-specific init function to
    *  perform its initialization
   */

   if ( usr_devInit( device ) )
      return (struct Device*)device;

   /*
    *  usr_devInit() says don't open the device.
    *  need to close dos library and return NULL.
   */

   if ( DOSBase )
      CloseLibrary( (struct Library*)DOSBase );

   DOSBase = NULL;
   return NULL;
}

/*=======================================================*/
/*
 *  dev_Open() - open a device
 *
 *  This procedure sets up the device for a calling
 *  process to use.  It is single-threaded by exec 
 *  (via Forbid()-Permit()), so it needs to return
 *  quickly.
 *
 *  This procedure is supposed to set up the IO_ERROR
 *  field, and the IO_UNIT and LN_TYPE fields unless
 *  an error exists.  Exec sets up the IO_DEVICE field.
 *
 *  NOTE:  a call to allocate memory in this routine
 *         can trigger a call to Expunge, so we need
 *         to fake an open count to prevent this if 
 *         we alocate memory
*/
/*=======================================================*/

long dev_Open( int unit,struct IOStdReq *devReq,
               ULONG flags, struct GenericDevice *device)
{
   /*
    *  fill out the io request with the device info
   */

   devReq->io_Device = (struct Device*)device;
   devReq->io_Unit   = 0;
   devReq->io_Flags  = 0;
   devReq->io_Error  = 0;

   /*
    *  call the device-specific open function
   */

   usr_devOpen( unit,devReq,flags,device );

   /*
    *  Unless an error occurred,
    *  increment the open count and clear any delayed
    *  expunges
   */
   
   if ( devReq->io_Error == 0 )
   {
      device->dev_Lib.lib_OpenCnt++;
      device->dev_Lib.lib_Flags &= ~LIBF_DELEXP;
   }

   /*
    *  return the error code - zero is success for
    *  devices and libs
   */

   return devReq->io_Error;
}

/*=======================================================*/
/*
 *  dev_Close() - close a device
 *
 *  This procedure is called to detach a device from a
 *  process that is using it. 
 *
 *  It can return one of two values:
 *
 *  NULL    - the device will not be unloaded
 *  seglist - the device will be unloaded
*/
/*=======================================================*/

BPTR dev_Close( struct IOStdReq *devReq,
                struct GenericDevice *device )
{
   BPTR segptr = NULL;

   /*
    *  call the device-specific close function to 
    *  perform any per task device cleanup
   */

   usr_devClose( devReq,device );

   /*
    *  set the unit and device numbers to -1 since this
    *  task is closing the device.  This is checked in
    *  BeginIO
    *  (see page 8, ramdev.device.asm, of RKM Devices)
   */

   devReq->io_Unit   = (struct Unit*)-1;
   devReq->io_Device = (struct Device*)-1;

   /*
    *  decrement open count & check for a delayed
    *  expunge.  If the flag is set, go ahead and
    *  call the expunge function.
   */

   if ( --device->dev_Lib.lib_OpenCnt == 0 )
   {
      if ( device->dev_Lib.lib_Flags & LIBF_DELEXP )
         segptr = dev_Expunge( device );
   }

   return segptr;
}

/*=======================================================*/
/*
 *  dev_Expunge() - remove a device from the system
 *
 *  The memory allocator calls the procedure when system
 *  is low on memory.  Since it is called by the memory
 *  allocator, it MUST return as quickly as possible.
 *
 *  It returns either:
 *
 *  segptr - if device no longer open and can be removed
 *  NULL   - set delayed expunge flag and return this.
*/
/*=======================================================*/

BPTR dev_Expunge( struct GenericDevice *device )
{
   BPTR segptr = NULL;
   char *addr;
   int  nb;

   /*
    *  see if anyone has the device open
    *  if so, set delayed expunge flag and return
   */

   if ( device->dev_Lib.lib_OpenCnt )
   {
      device->dev_Lib.lib_Flags |= LIBF_DELEXP;
      return NULL;
   }

   /*
    *  No one using us.
    *
    *  Call device-specific expunge procedure.  
    *  If it returns 0, we will not remove the device.  
    *  If it returns a non-zero value, we will remove
    *  the device.
   */

   if ( usr_devExpunge( device ) == 0 )
   {
      device->dev_Lib.lib_Flags |= LIBF_DELEXP;
      return NULL;
   }

   /*
    *  No one using us and the user procedure says it
    *  is ok to expunge.
    *  get rid of the device 
    *  First, close Dos library and remove the
    *  device from the system
   */

   if ( DOSBase )
      CloseLibrary( (struct Library*)DOSBase );

   DOSBase = NULL;

   segptr = (BPTR)device->dev_Data.segList;
   Remove((struct Node*)device);

   /*
    *  Free the device memory - need the base address
    *  and the total size of the device.  These values
    *  are calculated from LIB_POSSIZE & LIB_NEGSIZE
   */

   addr = (char*)device - device->dev_Lib.lib_NegSize;
   nb   = device->dev_Lib.lib_NegSize +
          device->dev_Lib.lib_PosSize;

   FreeMem(addr,nb);
   
   return segptr;
}

/*=======================================================*/
/*
 *  dev_BeginIO() - process device io commands.
 *
 *  This procedure is called from a client task (one that
 *  opened the device).  It is multi-threaded, so it 
 *  needs to take care of locking its resources as
 *  necessary.
*/
/*=======================================================*/

void dev_BeginIO( struct IOStdReq *devReq )
{
   struct GenericDevice *dev = 
            (struct GenericDevice*)devReq->io_Device;
   int reply = 0;

   /*
    *  So that WaitIO will work correctly, we need to set
    *  the ln_type to NT_MESSAGE.  This is set to 
    *  NT_REPLYMSG when the request is complete.
    *
    *  This procedure is multi-threaded.  If the device
    *  is managing a single global data segment, it will
    *  need to lock out other tasks while it is processing
    *  the device command.  This can be done with
    *  an exec signal semaphore.
    *
    *  This procedure either returns to the calling task,
    *  or replies to the calling task's message, depending
    *  on whether or not the command was completed 
    *  immediately or not.  If a command can be completed
    *  immediately, this procedure leaves the QUICK_IO
    *  flag set and simply returns.  The calling task can
    *  check this flag to determine if it needs to wait
    *  for a reply message from the device.
    *
    *  If the command can not be completed immediately,
    *  the device clears the QUICK_IO flag to let the
    *  caller know the device will send it a reply at a
    *  later time, and returns.
    *  
   */

   devReq->io_Message.mn_Node.ln_Type = NT_MESSAGE;
   devReq->io_Error = 0;

   /*
    *  If debug is on, write trace message
   */

   if ( dev->dev_Data.dbgFH )
   {
      FPrintf(dev->dev_Data.dbgFH,">> dev_BeginIO:  ");
      FPrintf(dev->dev_Data.dbgFH,
              "processing command %ld\n",
               devReq->io_Command);
   }

   /*
    *  call the user io procedure.  If it returns 1 and the
    *  QUICK_IO flag is clear, ReplyMsg the request.  
    *  Otherwise, unlock the device and return.
   */

   reply = usr_devBeginIO( devReq );  
   if ( reply )
   {
      if ( devReq->io_Flags & IOF_QUICK )
         ReplyMsg((struct Message*)devReq);
   }

   return;
}

/*=======================================================*/
/*
 *  dev_AbortIO() - abort device io commands
 *
 *  This procedure is multi-threaded.  It is called by
 *  tasks that have (or suspect they have) io requests
 *  that have not been processed by the device.  For
 *  example, a read command could be queued by the device
 *  waiting for a write from another task.  This procedure
 *  removes any io requests that are pending for the
 *  calling task and replies to the caller.
 *
*/
/*=======================================================*/

void dev_AbortIO( struct IOStdReq *devReq )
{
   struct GenericDevice *dev = 
            (struct GenericDevice*)devReq->io_Device;

   /*
    *  if debug is on, print trace message
   */

   if ( dev->dev_Data.dbgFH )
   {
      FPrintf(dev->dev_Data.dbgFH,">> dev_AbortIO...\n");
   }

   /*
    *  call the device-specific abort procedure to
    *  take care of any per-task cleanup
   */

   usr_devAbortIO( devReq );

   /*
    *  set io error return to IOERR_ABORTED (defined in 
    *  exec/errors.h) and reply to the message.
   */

   devReq->io_Error = IOERR_ABORTED; 
   ReplyMsg( (struct Message*)devReq );

   return;
}

